<?php

class Widget_API_Client
{

    private $token;
    private $base_url;

    public function __construct($token, $base_url = PX_WIDGET_API_URL)
    {
        $this->token = $token;
        $this->base_url = rtrim($base_url, '/');
    }

    private function get_headers()
    {
        return [
            'token' => $this->token,
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
        ];
    }

    private function request($path, $method = 'GET', $body = null, $query = [], $return_raw = false)
    {
        $url = $this->base_url . '/' . ltrim($path, '/');

        if (!empty($query)) {
            $url = add_query_arg($query, $url);
        }

        $args = [
            'method' => $method,
            'headers' => $this->get_headers(),
            'timeout' => 30,
        ];

        if ($body !== null) {
            $args['body'] = json_encode($body);
        }

        $response = wp_remote_request($url, $args);

        if (is_wp_error($response)) {
            return $response;
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if ($return_raw) {
            return $data;
        }

        if (is_array($data)) {
            if (isset($data['error']) && $data['error'] === true) {
                return new WP_Error('api_error', $data['message'] ?? 'Unknown API error');
            }

            unset($data['status'], $data['message']);

            if (isset($data['body'])) {
                return $data['body'];
            }

            if (count($data) === 1) {
                return reset($data);
            }
        }

        return $data;
    }

    public function get_carriers()
    {
        return $this->request('paxyWidget/carriers');
    }

    public function get_countries()
    {
        return $this->request('paxyWidget/countries');
    }

    public function parse_point_id($id)
    {
        return $this->request('paxyWidget/parsePointId', 'POST', ['id' => $id]);
    }

    public function find_point($carrier, $id_point)
    {
        return $this->request("paxyWidget/points/{$carrier}/{$id_point}");
    }

    public function list_points($carrier, $args = [])
    {
        $query = array_intersect_key($args, array_flip(['search', 'lat', 'lng', 'limit', 'offset']));
        return $this->request("paxyWidget/points/{$carrier}", 'GET', null, $query, true);
    }
}
