<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Paxy Shipping Method.
 */
class PX_Shipping_Method extends WC_Shipping_Method
{
    /**
     * Constructor.
     *
     * @param int $instance_id Instance ID.
     */
    public function __construct($instance_id = 0)
    {
        $this->id = 'paxy_shipping';
        $this->instance_id = absint($instance_id);

        // Fallback for some AJAX requests where instance_id might not be passed to constructor
        if (0 === $this->instance_id && is_admin() && isset($_REQUEST['instance_id'])) {
            $this->instance_id = absint($_REQUEST['instance_id']);
        }

        $this->method_title = __('Paxy', 'paxy');
        $this->method_description = __('Shipping method via Paxy.', 'paxy');
        $this->supports = [
            'shipping-zones',
            'instance-settings',
            'instance-settings-modal',
        ];

        $this->init();
    }

    /**
     * Initialize settings.
     */
    public function init()
    {
        // Load the settings.
        $this->init_form_fields();
        $this->init_settings();

        // Define user set variables.
        $this->title = $this->get_option('title');

        // Actions.
        add_action('woocommerce_update_options_shipping_' . $this->id, [$this, 'process_admin_options']);
    }

    /**
     * Init form fields.
     */
    public function init_form_fields()
    {
        $api_client = Paxy::get_api_client();
        $available_countries = $api_client->get_countries();
        $zone_countries = $this->get_zone_countries();
        $countries_options = ['' => __('Select a country', 'paxy')];

        $wc_countries = WC()->countries->get_countries();
        if (is_array($available_countries)) {
            foreach ($available_countries as $country) {
                if (in_array($country['code'], $zone_countries)) {
                    $countries_options[$country['id']] = $wc_countries[$country['code']] ?? $country['code'];
                }
            }
        }

        $selected_country_id = $this->get_option('country');
        if (empty($selected_country_id)) {
            $selected_country_id = $this->get_instance_option('country');
        }

        $carriers_options = ['' => __('Select a carrier', 'paxy')];

        if (!empty($selected_country_id)) {
            $carriers = $api_client->get_carriers_for_country($selected_country_id);
            if (is_array($carriers)) {
                foreach ($carriers as $carrier) {
                    $carriers_options[$carrier['id']] = $carrier['name'];
                }
            }
        }

        $selected_carrier_id = $this->get_option('carrier');
        if (empty($selected_carrier_id)) {
            $selected_carrier_id = $this->get_instance_option('carrier');
        }

        $package_type_options = ['' => __('Select a package type', 'paxy')];

        if (!empty($selected_country_id) && !empty($selected_carrier_id)) {
            $dimensions = $api_client->get_dimensions($selected_country_id, $selected_carrier_id);
            if (is_array($dimensions)) {
                foreach ($dimensions as $dimension) {
                    $package_type_options[$dimension['id']] = $dimension['label'];
                }
            }
        }

        $widget_client = Paxy::get_widget_client();
        $widget_carriers = $widget_client->get_carriers();
        $pickup_point_carriers_options = ['' => __('None', 'paxy')];

        if (is_array($widget_carriers)) {
            foreach ($widget_carriers as $index => $widget_carrier) {
                $pickup_point_carriers_options[$widget_carrier] = $widget_carrier;
            }
        }

        $this->instance_form_fields = [
            'title' => [
                'title' => __('Title', 'paxy'),
                'type' => 'text',
                'description' => __('This controls the title which the user sees during checkout.', 'paxy'),
                'desc_tip' => true,
                'required' => true,
            ],
            'cost' => [
                'title' => __('Cost', 'paxy'),
                'type' => 'text',
                'placeholder' => '0',
                'description' => __('Cost of delivery.', 'paxy'),
                'default' => '0',
                'desc_tip' => true,
            ],
            'country' => [
                'title' => __('Country', 'paxy'),
                'type' => 'select',
                'default' => '',
                'options' => $countries_options,
                'description' => __('Select which country this method is available for.', 'paxy'),
                'desc_tip' => true,
                'required' => true,
            ],
            'carrier' => [
                'title' => __('Carrier', 'paxy'),
                'type' => 'select',
                'default' => '',
                'options' => $carriers_options,
                'description' => __('Select which carrier handle this method.', 'paxy'),
                'desc_tip' => true,
                'required' => true,
            ],
            'package_type' => [
                'title' => __('Package Type', 'paxy'),
                'type' => 'select',
                'default' => '',
                'options' => $package_type_options,
                'description' => __('Select which package type this method support.', 'paxy'),
                'desc_tip' => true,
                'required' => true,
            ],
            'pickup_point' => [
                'title' => __('Pickup Point', 'paxy'),
                'type' => 'select',
                'default' => '',
                'options' => $pickup_point_carriers_options,
                'description' => __('If this method is for a pickup point, select the widget carrier.', 'paxy'),
                'desc_tip' => true,
            ],
        ];

        $this->form_fields = $this->instance_form_fields;
    }

    /**
     * Calculate shipping.
     *
     * @param array $package Package information.
     */
    public function calculate_shipping($package = [])
    {
        $cost = $this->get_option('cost');

        $rate = [
            'id' => $this->get_rate_id(),
            'label' => $this->title,
            'cost' => $cost,
            'package' => $package,
        ];

        $this->add_rate($rate);
    }

    /**
     * Get countries of the shipping zone.
     *
     * @return array
     */
    public function get_zone_countries()
    {
        $shipping_zone = WC_Shipping_Zones::get_zone_by('instance_id', $this->instance_id);
        $locations = $shipping_zone->get_zone_locations();
        $countries = [];

        foreach ($locations as $location) {
            if ('country' === $location->type) {
                $countries[] = $location->code;
            }
        }

        return $countries;
    }

    /**
     * Validate the title field.
     *
     * @param string $key Field key.
     * @param string $value Field value.
     * @return string
     */
    public function validate_title_field($key, $value)
    {
        if (empty($value)) {
            $this->add_error(__('Please enter a title.', 'paxy'));
        }

        return $value;
    }

    /**
     * Validate the country field.
     *
     * @param string $key Field key.
     * @param string $value Field value.
     * @return string
     */
    public function validate_country_field($key, $value)
    {
        if (empty($value)) {
            $this->add_error(__('Please select a country.', 'paxy'));
        }

        return $value;
    }

    /**
     * Validate the carrier field.
     *
     * @param string $key Field key.
     * @param string $value Field value.
     * @return string
     */
    public function validate_carrier_field($key, $value)
    {
        if (empty($value)) {
            $this->add_error(__('Please select a carrier.', 'paxy'));
        }

        return $value;
    }

    /**
     * Validate the package_type field.
     *
     * @param string $key Field key.
     * @param string $value Field value.
     * @return string
     */
    public function validate_package_type_field($key, $value)
    {
        if (empty($value)) {
            $this->add_error(__('Please select a package type.', 'paxy'));
        }

        return $value;
    }
}
