<?php

class Paxy
{
    private static $initiated = false;

    public static function plugin_activation()
    {
        if (version_compare($GLOBALS['wp_version'], PX__MINIMUM_WP_VERSION, '<')) {
            $message = '<strong>' . sprintf(esc_html__('Paxy %s requires WordPress %s or higher.', 'sq'), PX_VERSION, PX__MINIMUM_WP_VERSION) . '</strong> ' . sprintf(__('Please <a href="%1$s">upgrade WordPress</a> to a current version.', 'sq'), 'https://codex.wordpress.org/Upgrading_WordPress');
            self::display_activation_message($message);
        }
        add_option('px_db_version', PX_DB_VERSION);
    }

    public static function plugin_deactivation()
    {
        /* None */
    }

    public static function init()
    {
        if (!self::$initiated) {
            self::init_hook();
        }
    }

    private static function init_hook()
    {
        self::$initiated = true;

        load_plugin_textdomain('paxy', false, dirname(PX_BASENAME) . '/languages');

        add_action('wp_enqueue_scripts', [self::class, 'enqueue_scripts']);
        add_filter('woocommerce_shipping_methods', [self::class, 'register_shipping_method']);
        add_action('woocommerce_checkout_update_order_meta', [self::class, 'save_pickup_point_to_order'], 10, 2);
        add_action('woocommerce_checkout_create_order', [self::class, 'save_pickup_point_to_order'], 10, 2);
        add_action('woocommerce_store_api_checkout_update_order_from_request', [self::class, 'save_pickup_point_from_store_api'], 10, 2);
        add_action('woocommerce_blocks_loaded', [self::class, 'register_store_api_schema']);
        add_action('wp_ajax_px_get_points_for_widget', [self::class, 'ajax_get_points_for_widget']);
        add_action('wp_ajax_nopriv_px_get_points_for_widget', [self::class, 'ajax_get_points_for_widget']);
    }

    public static function register_store_api_schema()
    {
        if (function_exists('woocommerce_store_api_register_endpoint_data')) {
            woocommerce_store_api_register_endpoint_data(
                [
                    'endpoint' => 'checkout',
                    'namespace' => 'paxy-shipping',
                    'data_callback' => function () {
                        return [];
                    },
                    'schema_callback' => function () {
                        return [
                            'paxy_pickup_point_id' => ['type' => 'string', 'required' => false, 'sanitize_callback' => 'sanitize_text_field'],
                            'paxy_pickup_point_carrier' => ['type' => 'string', 'required' => false, 'sanitize_callback' => 'sanitize_text_field'],
                            'paxy_pickup_point_details' => ['type' => 'string', 'required' => false, 'sanitize_callback' => 'sanitize_text_field'],
                        ];
                    },
                    'schema_type' => ARRAY_A,
                ]
            );
            error_log('Paxy: Schema registered successfully via woocommerce_store_api_register_endpoint_data');
        } else {
            error_log('Paxy: woocommerce_store_api_register_endpoint_data function not found');
        }
    }

    public static function register_shipping_method($methods)
    {
        require_once(PX__PLUGIN_DIR . 'class.shipping-method.php');
        $methods['paxy_shipping'] = 'PX_Shipping_Method';
        return $methods;
    }

    public static function enqueue_scripts()
    {
        if (function_exists('is_checkout') && is_checkout()) {
            wp_enqueue_style('paxy-core', PX__PLUGIN_URL . 'assets/css/paxywidget.css', [], PX_VERSION);
            wp_enqueue_style('paxy-front', PX__PLUGIN_URL . 'assets/css/paxy-front.css', ['paxy-core'], PX_VERSION);
            wp_enqueue_script('paxy-core', PX__PLUGIN_URL . 'assets/js/paxywidget.js', [], PX_VERSION, true);
            wp_enqueue_script('paxy-front', PX__PLUGIN_URL . 'assets/js/paxy-front.js', ['jquery', 'paxy-core'], PX_VERSION, true);

            // Localize data for JS to handle Blocks/dynamic changes
            wp_localize_script('paxy-front', 'paxyData', [
                'shippingMethods' => self::get_shipping_methods_data(),
                'choosePointLabel' => __('Choose pickup point', 'paxy'),
                'selectedPointLabel' => __('Selected pickup point', 'paxy'),
                'widgetApiUrl' => admin_url('admin-ajax.php?action=px_get_points_for_widget'),
                'widgetAssetsUrl' => PX__PLUGIN_URL,
                'widgetLang' => substr(get_locale(), 0, 2)
            ]);
        }
    }

    private static function get_shipping_methods_data()
    {
        $data = [];
        $delivery_zones = WC_Shipping_Zones::get_zones();
        // Add default zone
        $delivery_zones[] = [
            'zone_id' => 0,
            'shipping_methods' => WC_Shipping_Zones::get_zone(0)->get_shipping_methods()
        ];

        foreach ($delivery_zones as $zone) {
            foreach ($zone['shipping_methods'] as $method) {
                if ('paxy_shipping' === $method->id) {
                    $instance_id = $method->instance_id;
                    $pickup_point = $method->get_option('pickup_point');
                    if (!empty($pickup_point)) {
                        $data['paxy_shipping:' . $instance_id] = [
                            'carrier' => $pickup_point,
                            'instanceId' => $instance_id
                        ];
                    }
                }
            }
        }
        return $data;
    }

    public static function save_pickup_point_to_order($order_id_or_order, $data)
    {
        $order = null;
        if (is_a($order_id_or_order, 'WC_Order')) {
            $order = $order_id_or_order;
        } else {
            $order = wc_get_order($order_id_or_order);
        }

        if (!$order) {
            return;
        }

        if (isset($_POST['paxy_pickup_point_id']) && !empty($_POST['paxy_pickup_point_id'])) {
            $order->update_meta_data('_paxy_pickup_point_id', sanitize_text_field($_POST['paxy_pickup_point_id']));
        }
        if (isset($_POST['paxy_pickup_point_carrier']) && !empty($_POST['paxy_pickup_point_carrier'])) {
            $order->update_meta_data('_paxy_pickup_point_carrier', sanitize_text_field($_POST['paxy_pickup_point_carrier']));
        }
        if (isset($_POST['paxy_pickup_point_details']) && !empty($_POST['paxy_pickup_point_details'])) {
            $order->update_meta_data('_paxy_pickup_point_details', sanitize_text_field($_POST['paxy_pickup_point_details']));
        }

        if (is_a($order_id_or_order, 'WC_Order')) {
            // No need to save if it's the object, it will be saved by WC
        } else {
            $order->save();
        }
    }

    public static function save_pickup_point_from_store_api($order, $request)
    {
        $data = $request->get_param('extensions');

        if (isset($data['paxy-shipping']) && is_array($data['paxy-shipping'])) {
            $paxy_data = $data['paxy-shipping'];

            if (isset($paxy_data['paxy_pickup_point_id']) && !empty($paxy_data['paxy_pickup_point_id'])) {
                $order->update_meta_data('_paxy_pickup_point_id', sanitize_text_field($paxy_data['paxy_pickup_point_id']));
            }
            if (isset($paxy_data['paxy_pickup_point_carrier']) && !empty($paxy_data['paxy_pickup_point_carrier'])) {
                $order->update_meta_data('_paxy_pickup_point_carrier', sanitize_text_field($paxy_data['paxy_pickup_point_carrier']));
            }
            if (isset($paxy_data['paxy_pickup_point_details']) && !empty($paxy_data['paxy_pickup_point_details'])) {
                $order->update_meta_data('_paxy_pickup_point_details', sanitize_text_field($paxy_data['paxy_pickup_point_details']));
            }

            // Note: In Store API, the order is saved automatically after this hook
        } else {
            error_log('Paxy Store API Data: No paxy-shipping data found in extensions.');
        }
    }

    public static function ajax_get_points_for_widget()
    {
        $courier = sanitize_text_field($_GET['courier'] ?? '');
        if (empty($courier)) {
            wp_send_json_error('Missing courier');
        }

        $widget_client = self::get_widget_client();
        $points = $widget_client->list_points($courier, $_GET);

        if (is_wp_error($points)) {
            wp_send_json_error($points->get_error_message());
        }

        wp_send_json($points);
    }

    /**
     * @return Widget_API_Client
     */
    public static function get_widget_client()
    {
        require_once(PX__PLUGIN_DIR . 'class.widget-api-client.php');
        $options = get_option('paxy_options');
        $token = $options['widget_token'] ?? '';
        return new Widget_API_Client($token);
    }

    /**
     * @return Paxy_API_Client
     */
    public static function get_api_client()
    {
        require_once(PX__PLUGIN_DIR . 'class.paxy-api-client.php');
        $options = get_option('paxy_options');
        $key = $options['api_key'] ?? '';
        $token = $options['api_token'] ?? '';
        return new Paxy_API_Client($key, $token);
    }

    private static function display_activation_message($message)
    {
        ?>
        <!doctype html>
        <html>

        <head>
            <meta charset="<?php bloginfo('charset'); ?>">
            <style type="text/css">
                * {
                    margin: 0;
                    padding: 0;
                    font-family: 'Open Sans', sans-serif;
                }

                p {
                    font-size: 13px;
                    height: 19px;
                }
            </style>
        </head>

        <body>
            <p><?php echo $message; ?></p>
        </body>

        </html>
        <?php
        exit;
    }
}
