<?php

class Paxy_API_Client
{

    private $api_key;
    private $api_token;
    private $base_url;

    public function __construct($api_key, $api_token, $base_url = PX_API_URL)
    {
        $this->api_key = $api_key;
        $this->api_token = $api_token;
        $this->base_url = rtrim($base_url, '/');
    }

    private function get_headers()
    {
        return [
            'cl-api-key' => $this->api_key,
            'cl-api-token' => $this->api_token,
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
        ];
    }

    private function request($path, $method = 'GET', $body = null, $query = [])
    {
        $url = $this->base_url . '/' . ltrim($path, '/');

        if (!empty($query)) {
            $url = add_query_arg($query, $url);
        }

        $args = [
            'method' => $method,
            'headers' => $this->get_headers(),
            'timeout' => 30,
        ];

        if ($body !== null) {
            $args['body'] = json_encode($body);
        }

        $response = wp_remote_request($url, $args);

        if (is_wp_error($response)) {
            return $response;
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        // For labels, we might get PDF content or direct response
        $content_type = wp_remote_retrieve_header($response, 'content-type');
        if (strpos($content_type, 'application/pdf') !== false) {
            return $response;
        }

        $data = json_decode($body, true);

        if ($response_code >= 400) {
            return new WP_Error('api_error', $data['message'] ?? 'API error ' . $response_code);
        }

        if (is_array($data)) {
            unset($data['status'], $data['message']);

            // If there's only one key left, return its value to further simplify
            if (count($data) === 1) {
                return reset($data);
            }
        }

        return $data;
    }

    public function get_countries()
    {
        return $this->request('country/');
    }

    public function get_country($id)
    {
        return $this->request("country/{$id}");
    }

    public function get_carrier($id)
    {
        return $this->request("carrier/{$id}");
    }

    public function get_carriers_for_country($country_id)
    {
        return $this->request("carrier/country/{$country_id}");
    }

    public function get_dimensions($country_id, $carrier_id)
    {
        return $this->request('dimension/', 'GET', null, [
            'countryId' => $country_id,
            'carrierId' => $carrier_id,
        ]);
    }

    public function create_book($data)
    {
        return $this->request('ksiazka/utworz', 'POST', $data);
    }

    public function add_parcel($data)
    {
        return $this->request('przesylki/dodaj', 'POST', $data);
    }

    public function get_label($args)
    {
        // This is usually a GET request that returns PDF
        return $this->request('etykieta/drukuj', 'GET', null, $args);
    }
}
