<?php

class PaxyAdmin
{
    private static $initiated = false;

    public static function init()
    {
        if (!self::$initiated) {
            self::init_hook();
        }
    }

    private static function init_hook()
    {
        add_action('admin_menu', [self::class, 'add_admin_menu']);
        add_action('admin_init', [self::class, 'add_settings']);
        add_action('admin_enqueue_scripts', [self::class, 'enqueue_scripts']);
        add_filter('plugin_action_links_' . PX_BASENAME, [self::class, 'add_action_links']);
        add_action('add_meta_boxes', [self::class, 'add_order_meta_box']);
        add_action('admin_notices', [self::class, 'display_missing_settings_notice']);
        add_action('wp_ajax_px_get_carriers', [self::class, 'ajax_get_carriers']);
        add_action('wp_ajax_px_get_dimensions', [self::class, 'ajax_get_dimensions']);
        add_action('wp_ajax_px_create_parcel', [self::class, 'ajax_create_parcel']);
        add_action('wp_ajax_px_download_label', [self::class, 'ajax_download_label']);

        self::$initiated = true;
    }

    public static function add_admin_menu()
    {
        if (!current_user_can('manage_options')) {
            return;
        }

        add_submenu_page('options-general.php', 'Paxy', 'Paxy', 'manage_options', 'paxy-settings', [self::class, 'render_settings']);
    }

    public static function add_settings()
    {
        register_setting('paxy', 'paxy_options');

        add_settings_section(
            'paxy_api',
            __('Api'),
            [self::class, 'render_section'],
            'paxy'
        );

        add_settings_field(
            'api_key',
            __('Api key'),
            [self::class, 'render_api_key_field'],
            'paxy',
            'paxy_api'
        );

        add_settings_field(
            'api_token',
            __('Api token'),
            [self::class, 'render_api_token_field'],
            'paxy',
            'paxy_api'
        );

        add_settings_section(
            'paxy_widget',
            __('Widget'),
            [self::class, 'render_section'],
            'paxy'
        );

        add_settings_field(
            'widget_token',
            __('Widget api token'),
            [self::class, 'render_widget_api_token_field'],
            'paxy',
            'paxy_widget'
        );
    }

    public static function render_settings()
    {
        ?>
        <div class="wrap">
            <form action='options.php' method='post'>
                <h1><?= __('Settings'); ?></h1>
                <?php
                settings_fields('paxy');
                do_settings_sections('paxy');
                submit_button();
                ?>
            </form>
        </div>
        <?php
    }

    public static function render_section()
    {

    }

    public static function render_api_key_field()
    {
        $options = get_option('paxy_options');
        ?>
        <input type="password" name="paxy_options[api_key]" class="regular-text" <?php if (isset($options['api_key']) && $options['api_key'] !== '') { ?>value="<?= $options['api_key']; ?>" <?php } ?>>
        <?php
    }

    public static function render_api_token_field()
    {
        $options = get_option('paxy_options');
        ?>
        <input type="password" name="paxy_options[api_token]" class="regular-text" <?php if (isset($options['api_token']) && $options['api_token'] !== '') { ?>value="<?= $options['api_token']; ?>" <?php } ?>>
        <?php
    }

    public static function render_widget_api_token_field()
    {
        $options = get_option('paxy_options');
        ?>
        <input type="password" name="paxy_options[widget_token]" class="regular-text" <?php if (isset($options['widget_token']) && $options['widget_token'] !== '') { ?>value="<?= $options['widget_token']; ?>" <?php } ?>>
        <?php
    }

    public static function enqueue_scripts($hook)
    {
        $screen = get_current_screen();
        $is_order_page = $screen && ($screen->id === 'shop_order' || $screen->id === 'woocommerce_page_wc-orders' || ($screen->base === 'post' && $screen->post_type === 'shop_order'));
        $is_settings_page = $hook === 'woocommerce_page_wc-settings';

        if ($is_order_page || $is_settings_page) {
            wp_enqueue_style('paxy-admin', PX__PLUGIN_URL . 'assets/css/paxy-admin.css', [], PX_VERSION);

            wp_enqueue_script('paxy-admin', PX__PLUGIN_URL . 'assets/js/paxy-admin.js', ['jquery'], PX_VERSION, true);
            wp_localize_script('paxy-admin', 'paxy_admin', [
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('paxy_admin_nonce'),
            ]);

            wp_localize_script('paxy-admin', 'paxy_admin_i18n', [
                'select_country_first' => __('Select a country first', 'paxy'),
                'select_carrier_first' => __('Select a carrier first', 'paxy'),
                'loading_carriers' => __('Loading carriers...', 'paxy'),
                'loading_types' => __('Loading package types...', 'paxy'),
                'select_carrier' => __('Select a carrier', 'paxy'),
                'select_type' => __('Select a package type', 'paxy'),
                'required' => __('is required!', 'paxy'),
                'weight_required' => __('Weight is required!', 'paxy'),
                'quantity_required' => __('Quantity is required!', 'paxy'),
                'parcel_generate_success' => __('Parcel created successfully!', 'paxy'),
                'parcel_generate_error' => __('An error occurred while creating the parcel.', 'paxy'),
                'generic_error' => __('Error', 'paxy'),
            ]);
        }
    }

    public static function add_action_links($links)
    {
        $settings_link = '<a href="' . admin_url('options-general.php?page=paxy-settings') . '">' . __('Settings', 'paxy') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }

    public static function add_order_meta_box()
    {
        $screen = class_exists('Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController') &&
            \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled()
            ? wc_get_page_screen_id('shop-order')
            : 'shop_order';

        add_meta_box(
            'paxy-order-details',
            esc_html__('Paxy', 'paxy'),
            [self::class, 'display_pickup_point_in_admin'],
            $screen,
            'normal',
            'high'
        );
    }

    public static function display_pickup_point_in_admin($post)
    {
        $order = ($post instanceof WP_Post) ? wc_get_order($post->ID) : $post;
        $shipping_methods = $order->get_shipping_methods();
        $is_paxy = false;
        $instance_id = 0;
        foreach ($shipping_methods as $method) {
            if (strpos($method->get_method_id(), 'paxy_shipping') !== false) {
                $is_paxy = true;
                $instance_id = $method->get_instance_id();
                break;
            }
        }

        if (!$is_paxy) {
            echo '<div class="paxy-admin-pickup-point-container">
                <div class="paxy-pickup-point-info">
                    <p>' . esc_html__('This order is not processed by Paxy.', 'paxy') . '</p>
                </div>
            </div>';
            return;
        }

        $shipping_method_details = '';
        if ($instance_id) {
            require_once(PX__PLUGIN_DIR . 'class.shipping-method.php');
            $shipping_method = new PX_Shipping_Method($instance_id);
            $c_id = $shipping_method->get_option('country');
            $cr_id = $shipping_method->get_option('carrier');
            $pt_id = $shipping_method->get_option('package_type');

            $country_name = $shipping_method->form_fields['country']['options'][$c_id] ?? $c_id;
            $carrier_name = $shipping_method->form_fields['carrier']['options'][$cr_id] ?? $cr_id;
            $package_type_name = $shipping_method->form_fields['package_type']['options'][$pt_id] ?? $pt_id;

            $shipping_method_details = '
                <p><strong>' . esc_html__('Country:', 'paxy') . '</strong> ' . esc_html($country_name) . '</p>
                <p><strong>' . esc_html__('Carrier:', 'paxy') . '</strong> ' . esc_html($carrier_name) . '</p>
                <p><strong>' . esc_html__('Package Type:', 'paxy') . '</strong> ' . esc_html($package_type_name) . '</p>
            ';
        }

        if (empty($shipping_method_details)) {
            return;
        }

        $point_id = $order->get_meta('_paxy_pickup_point_id');
        $point_carrier = $order->get_meta('_paxy_pickup_point_carrier');
        $point_details = $order->get_meta('_paxy_pickup_point_details');
        $parcel_number = $order->get_meta('_paxy_parcel_number');
        $parcel_created = $order->get_meta('_paxy_parcel_created');

        $ret = '<div class="paxy-admin-pickup-point-container">
            <div class="paxy-pickup-point-info">' . $shipping_method_details . '</div>';

        if (!empty($point_id) && !empty($point_carrier) && !empty($point_details)) {
            $ret .= '<div class="paxy-pickup-point-info">
                <p class="paxy-pickup-point-info-lead">' . esc_html__('Pickup Point', 'paxy') . '</p>
                <p style="display: none;"><strong>' . esc_html__('Carrier:', 'paxy') . '</strong> ' . esc_html($point_carrier) . '</p>
                <p><strong>' . esc_html__('Point ID:', 'paxy') . '</strong> ' . esc_html($point_id) . '</p>
                <p><strong>' . esc_html__('Details:', 'paxy') . '</strong> ' . esc_html($point_details) . '</p>
            </div>';
        }

        if ($parcel_number) {
            $ret .= '<div class="paxy-pickup-point-info">
                <p class="paxy-pickup-point-info-lead">' . esc_html__('Parcel', 'paxy') . '</p>
                <p><strong>' . esc_html__('Package number:', 'paxy') . '</strong> ' . esc_html($parcel_number) . '</p>
                <p><strong>' . esc_html__('Date of creation:', 'paxy') . '</strong> ' . esc_html($parcel_created) . '</p>
            </div>
            <button type="button" class="button button-primary" id="paxy-download-label" data-order-id="' . esc_attr($order->get_id()) . '">' . esc_html__('Download label', 'paxy') . '</button>';
        } else {
            $ret .= '<div class="paxy-label-form-container">
                <div class="paxy-pickup-point-info">
                    <p class="paxy-pickup-point-info-lead">' . esc_html__('Send a parcel', 'paxy') . '</p>
                </div>
                <div class="paxy-pickup-point-form">
                    <input type="hidden" name="order_id" value="' . esc_attr($order->get_id()) . '">
                    <input type="hidden" name="action" value="px_create_parcel">
                    <p>
                        <label for="paxy_weight">' . esc_html__('Weight (kg):', 'paxy') . '</label>
                        <input type="number" step="0.01" name="paxy_weight">
                    </p>
                    <p>
                        <label for="paxy_quantity">' . esc_html__('Quantity:', 'paxy') . '</label>
                        <input type="number" name="paxy_quantity" value="1" required>
                    </p>
                    <button type="button" class="button button-primary" id="paxy-create-parcel">' . esc_html__('Submit', 'paxy') . '</button>
                </div>
            </div>';
        }

        $ret .= '</div>';

        echo $ret;
    }

    public static function display_missing_settings_notice()
    {
        if (!current_user_can('manage_options')) {
            return;
        }

        static $displayed = false;
        if ($displayed) {
            return;
        }

        $options = get_option('paxy_options');
        $missing = [];

        if (empty($options['api_key'])) {
            $missing[] = __('Api key', 'paxy');
        }
        if (empty($options['api_token'])) {
            $missing[] = __('Api token', 'paxy');
        }
        if (empty($options['widget_token'])) {
            $missing[] = __('Widget token', 'paxy');
        }

        if (!empty($missing)) {
            $displayed = true;
            $settings_url = admin_url('options-general.php?page=paxy-settings');
            ?>
            <div class="notice notice-warning is-dismissible">
                <p>
                    <?php
                    printf(
                        __('Paxy plugin is not fully configured. Missing: %1$s. Please check your <a href="%2$s">API settings</a>.', 'paxy'),
                        implode(', ', $missing),
                        $settings_url
                    );
                    ?>
                </p>
            </div>
            <?php
        }
    }

    public static function ajax_get_carriers()
    {
        check_ajax_referer('paxy_admin_nonce', 'nonce');

        $country_id = isset($_GET['country_id']) ? sanitize_text_field($_GET['country_id']) : '';
        if (empty($country_id)) {
            wp_send_json_error('Missing country ID');
        }

        $api_client = Paxy::get_api_client();
        $carriers = $api_client->get_carriers_for_country($country_id);
        wp_send_json_success($carriers);
    }

    public static function ajax_get_dimensions()
    {
        check_ajax_referer('paxy_admin_nonce', 'nonce');

        $country_id = isset($_GET['country_id']) ? sanitize_text_field($_GET['country_id']) : '';
        $carrier_id = isset($_GET['carrier_id']) ? sanitize_text_field($_GET['carrier_id']) : '';

        if (empty($country_id) || empty($carrier_id)) {
            wp_send_json_error('Missing country or carrier ID');
        }

        $api_client = Paxy::get_api_client();
        $dimensions = $api_client->get_dimensions($country_id, $carrier_id);
        wp_send_json_success($dimensions);
    }

    public static function ajax_create_parcel()
    {
        check_ajax_referer('paxy_admin_nonce', 'nonce');

        $order_id = isset($_POST['order_id']) ? (int) $_POST['order_id'] : 0;
        $weight = (isset($_POST['paxy_weight']) && $_POST['paxy_weight'] !== '') ? (float) $_POST['paxy_weight'] : 0;
        $quantity = isset($_POST['paxy_quantity']) ? (int) $_POST['paxy_quantity'] : 1;

        if (!$order_id || $quantity <= 0) {
            wp_send_json_error('Missing or invalid required fields');
        }

        $order = wc_get_order($order_id);
        if (!$order) {
            wp_send_json_error('Order not found');
        }

        if ($weight <= 0) {
            $total_weight = 0;
            foreach ($order->get_items() as $item_id => $item) {
                $product = $item->get_product();
                if ($product) {
                    $item_weight = (float) $product->get_weight();
                    $total_weight += $item_weight * $item->get_quantity();
                }
            }
            $weight = $total_weight;
        }

        if ($weight <= 0) {
            wp_send_json_error('Weight is required and could not be determined from the order.');
        }

        $api_client = Paxy::get_api_client();

        $shipping_methods = $order->get_shipping_methods();
        $country = '';
        $carrier = '';
        $package = '';
        foreach ($shipping_methods as $method) {
            if (strpos($method->get_method_id(), 'paxy_shipping') !== false) {
                $instance_id = $method->get_instance_id();
                if ($instance_id) {
                    require_once(PX__PLUGIN_DIR . 'class.shipping-method.php');
                    $shipping_method = new PX_Shipping_Method($instance_id);
                    $c_id = $shipping_method->get_option('country');
                    $cr_id = $shipping_method->get_option('carrier');
                    $package = $shipping_method->get_option('package_type');

                    $countryResponse = $api_client->get_country($c_id);
                    if (!empty($countryResponse)) {
                        $country = $countryResponse['name'];
                    }
                    $carrierResponse = $api_client->get_carrier($cr_id);
                    if (!empty($carrierResponse)) {
                        $carrier = $carrierResponse['apiCode'];
                    }
                }
                break;
            }
        }

        if (empty($country)) {
            wp_send_json_error('Could not determine Paxy shipping country for this order.');
        }
        if (empty($carrier)) {
            wp_send_json_error('Could not determine Paxy shipping carrier for this order.');
        }

        $point_id = $order->get_meta('_paxy_pickup_point_id');
        $point_carrier = $order->get_meta('_paxy_pickup_point_carrier');

        $point_details = null;
        $parsed_point = null;
        if (!empty($point_id)) {
            $widget_client = Paxy::get_widget_client();
            $point_details = $widget_client->find_point($point_carrier, $point_id);
            $parsed_point = $widget_client->parse_point_id($point_id);
        }

        $fields = [
            'rodzaj' => $package,
            'kodpocztowy' => $order->get_shipping_postcode(),
            'nazwaodbiorcy' => trim($order->get_shipping_company() . ' ' . $order->get_shipping_first_name() . ' ' . $order->get_shipping_last_name()),
            'ulica' => trim($order->get_shipping_address_1() . ' ' . $order->get_shipping_address_2()),
            'nrbudynku' => '',
            'miejscowosc' => $order->get_shipping_city(),
            'nrtelefonu' => $order->get_billing_phone(),
            'waga' => (string) round(($weight / $quantity), 1),
            'przewoznik' => $carrier,
            'doc_wz' => (string) $order->get_id(),
            'referencje' => (string) $order->get_id(),
            'ilosc' => $quantity,
            'email' => $order->get_billing_email()
        ];

        if ($order->get_payment_method() === 'cod') {
            $fields['pobranie'] = round((float) $order->get_total(), 2);
        }

        if (!empty($point_id) && !empty($point_details)) {
            if (strpos($point_id, 'cp_balikovna') !== false || strpos($point_id, 'cp_napostu') !== false) {
                if (isset($point_details['city'])) {
                    $fields['miejscowosc'] = $point_details['city'];
                }
                $tmpPointId = str_replace(['cp_balikovna', 'cp_napostu'], '', $point_id);
                $fields['kodpocztowy'] = $tmpPointId;
            }
        }

        if (!empty($parsed_point)) {
            $pointType = $parsed_point['pointType'] ?? '';
            $parsedId = $parsed_point['id'] ?? '';
            $fields['rodzaj'] = $pointType;
            if ($pointType === 'office') {
                $fields['point_id'] = null;
                $fields['office_id'] = $parsedId;
            } else {
                $fields['point_id'] = $parsedId;
                $fields['office_id'] = null;
            }
        }

        $book_response = $api_client->create_book(['kraj' => $country]);
        if (is_wp_error($book_response)) {
            wp_send_json_error('Book Creation Error: ' . $book_response->get_error_message());
        }

        $book_number = $book_response['nr'] ?? $book_response['data']['nr'] ?? '';
        if (empty($book_number)) {
            wp_send_json_error('Could not obtain book number from API.');
        }

        $fields['nr_ksiazki'] = $book_number;

        $parcel_response = $api_client->add_parcel($fields);
        if (is_wp_error($parcel_response)) {
            wp_send_json_error('Parcel Addition Error: ' . $parcel_response->get_error_message());
        }

        $parcel_number = $parcel_response['nr'] ?? $parcel_response['data']['nr'] ?? '';
        $internal_number = $parcel_response['internalNr'] ?? $parcel_response['data']['internalNr'] ?? '';
        $tracking_link_number = $parcel_response['trackingLink'] ?? $parcel_response['data']['trackingLink'] ?? '';
        $return_label_number = $parcel_response['returnLabelNr'] ?? $parcel_response['data']['returnLabelNr'] ?? '';
        if (!empty($parcel_number)) {
            $order->update_meta_data('_paxy_parcel_number', sanitize_text_field($parcel_number));
            $order->update_meta_data('_paxy_parcel_internal_number', sanitize_text_field($internal_number));
            $order->update_meta_data('_paxy_parcel_tracking_link_number', sanitize_text_field($tracking_link_number));
            $order->update_meta_data('_paxy_parcel_label_number', sanitize_text_field($return_label_number));
            $order->update_meta_data('_paxy_parcel_created', current_time('mysql'));
            $order->save();
        }

        wp_send_json_success($parcel_response);
    }

    public static function ajax_download_label()
    {
        check_ajax_referer('paxy_admin_nonce', 'nonce');

        $order_id = isset($_GET['order_id']) ? (int) $_GET['order_id'] : 0;
        if (!$order_id) {
            wp_die('Missing order ID');
        }

        $order = wc_get_order($order_id);
        if (!$order) {
            wp_die('Order not found');
        }

        $parcel_number = $order->get_meta('_paxy_parcel_number');
        if (!$parcel_number) {
            wp_die('No parcel number found for this order');
        }

        $api_client = Paxy::get_api_client();
        $response = $api_client->get_label(['nr_przesylki' => $parcel_number]);

        if (is_wp_error($response)) {
            wp_die('API Error: ' . $response->get_error_message());
        }

        // The API client request method returns the full response if it's a PDF
        $body = wp_remote_retrieve_body($response);
        $content_type = wp_remote_retrieve_header($response, 'content-type');

        if (empty($body)) {
            wp_die('Empty label received from API');
        }

        header('Content-Type: ' . ($content_type ?: 'application/pdf'));
        header('Content-Disposition: attachment; filename="' . $parcel_number . '.pdf"');
        echo $body;
        exit;
    }
}
