<?php


namespace Vendor\CzechlogisticIntegration\Controller\Adminhtml\Czechlogistic;


use Magento\Backend\App\Action\Context;
use Magento\Framework\Controller\ResultFactory;
use Magento\Framework\Message\ManagerInterface;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Api\OrderManagementInterface;
use Vendor\CzechlogisticIntegration\Model\ResourceModel\CzechlogisticOrder\Collection;
use Magento\Framework\ObjectManagerInterface;

class MultiParcelsCreate extends \Magento\Backend\App\Action
{
    private $czechApiUrl = 'https://api.czechlogistic.pl';

    /**
     * @var OrderManagementInterface
     */
    protected $orderManagement;

    /**
     * @var ScopeConfigInterface
     */
    private $scopeConfig;
    /**
     * @var ManagerInterface
     */
    private $manager;
    /**
     * @var OrderRepositoryInterface
     */
    private $orderRepository;
    /**
     * @var \Magento\Framework\Api\SearchCriteriaBuilder
     */
    private $searchCriteriaBuilder;
    /**
     * @var ObjectManagerInterface
     */
    private $objectManager;

    public function __construct(Context $context, \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig, ManagerInterface $manager, OrderRepositoryInterface $orderRepository, \Magento\Framework\Api\SearchCriteriaBuilder $searchCriteriaBuilder, ObjectManagerInterface $objectManager)
    {
        parent::__construct($context);
        $this->scopeConfig = $scopeConfig;
        $this->manager = $manager;
        $this->orderRepository = $orderRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->objectManager = $objectManager;
    }

    public function execute()
    {
        $resultRedirect = $this->resultFactory->create(ResultFactory::TYPE_REDIRECT);
        $resultRedirect->setPath($this->_redirect->getRefererUrl());

        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter('entity_id', $this->_request->getParam('selected'), 'in')->create();
        $orders = $this->orderRepository->getList($searchCriteria)->getItems();

        $skippedIds = [];
        $errorIds = [];

        foreach ($orders as $order) {
            if ($order->getCzechlogisticParcelNumber()) {
                $skippedIds[] = $order->getIncrementId();
                continue;
            }

            $czechlogisticOrder = $this->_objectManager->create(Collection::class)->addFieldToFilter('order_id', $order->getEntityId());
            $czechlogisticOrder = $czechlogisticOrder->getFirstItem();

            $countryId = $czechlogisticOrder->getCzechlogisticCountryId();
            $carrierId = $czechlogisticOrder->getCzechlogisticCarrierId();
            $parcelTypeId = $czechlogisticOrder->getCzechlogisticParcelId();

            if (empty($countryId) || empty($carrierId) || empty($parcelTypeId)) {
                continue;
            }

            $countryDetails = $this->loadCountryDetails($countryId);
            $book = $this->createBook($countryDetails['name']);
            $bookNumber = $book->ksiazka->nr;

            $carrier = $this->loadCarrierDetails($carrierId)['carrier'];
            $carrierCode = $carrier['apiCode'];

            $resultRedirect = $this->resultFactory->create(ResultFactory::TYPE_REDIRECT);
            $resultRedirect->setPath($this->_redirect->getRefererUrl());

            $parcel = $this->createParcel($order, $bookNumber, $parcelTypeId, $carrierCode, 1, $czechlogisticOrder);
            if (isset($parcel->przesylka->nr)) {
                $order->setCzechlogisticParcelNumber($parcel->przesylka->nr);
                $order->setCzechlogisticCreated(new \DateTime());
                $order->save();
                $this->messageManager->addSuccessMessage('Utworzono przesyłkę dla id: '.$order->getIncrementId().', numer:'. $parcel->przesylka->nr);
            }   else {
                $this->messageManager->addErrorMessage('Nie udało się utworzyć przesyłki dla id: '. $order->getIncrementId());
            }
        }

        if (count($skippedIds) > 0) {
            $this->messageManager->addWarningMessage('Pominięte zamówienia, ze względu na istniejący numer: '. implode(', ', $skippedIds));
        }

        return $resultRedirect;
    }


    private function createBook($countryName)
    {
        $apiKey = $this->scopeConfig->getValue('czechlogistic_config/general/apiKey', \Magento\Store\Model\ScopeInterface::SCOPE_STORE);
        $apiSecret = $this->scopeConfig->getValue('czechlogistic_config/general/apiSecret', \Magento\Store\Model\ScopeInterface::SCOPE_STORE);

        $ch = curl_init();
        $fields = array( 'kraj' => $countryName);
        $postvars = '';
        foreach($fields as $key=>$value) {
            $postvars .= $key . "=" . $value . "&";
        }
        $url = $this->czechApiUrl . '/ksiazka/utworz';
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            "CL-API-KEY: $apiKey",
            "CL-API-TOKEN: $apiSecret"
        ]);
        curl_setopt($ch,CURLOPT_URL,$url);
        curl_setopt($ch,CURLOPT_POST, 1);                //0 for a get request
        curl_setopt($ch,CURLOPT_POSTFIELDS,$postvars);
        curl_setopt($ch,CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch,CURLOPT_CONNECTTIMEOUT ,3);
        curl_setopt($ch,CURLOPT_TIMEOUT, 20);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, FALSE);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, FALSE);
        $response = curl_exec($ch);

        return json_decode($response);
    }

    private function loadCountryDetails($countryId)
    {
        $apiKey = $this->scopeConfig->getValue('czechlogistic_config/general/apiKey', \Magento\Store\Model\ScopeInterface::SCOPE_STORE);
        $apiSecret = $this->scopeConfig->getValue('czechlogistic_config/general/apiSecret', \Magento\Store\Model\ScopeInterface::SCOPE_STORE);

        $ch = curl_init();
        $endpoint = $this->czechApiUrl . '/country/'.$countryId;
        $params = array();
        $url = $endpoint . '?' . http_build_query($params);
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            "CL-API-KEY: $apiKey",
            "CL-API-TOKEN: $apiSecret",
            'Accept: application/json'
        ]);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, FALSE);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, FALSE);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $apiResponse = curl_exec($ch);
        curl_close($ch);
        if ($apiResponse) {
            $apiResponse = json_decode($apiResponse,true);
            return $apiResponse['country'];
        }

        return [];
    }

    private function loadCarrierDetails($carrierId)
    {
        $apiKey = $this->scopeConfig->getValue('czechlogistic_config/general/apiKey', \Magento\Store\Model\ScopeInterface::SCOPE_STORE);
        $apiSecret = $this->scopeConfig->getValue('czechlogistic_config/general/apiSecret', \Magento\Store\Model\ScopeInterface::SCOPE_STORE);

        $ch = curl_init();
        $endpoint = $this->czechApiUrl . '/carrier/'.$carrierId;
        $params = array();
        $url = $endpoint . '?' . http_build_query($params);
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            "CL-API-KEY: $apiKey",
            "CL-API-TOKEN: $apiSecret",
            'Accept: application/json'
        ]);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, FALSE);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, FALSE);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $apiResponse = curl_exec($ch);
        curl_close($ch);
        if ($apiResponse) {
            return json_decode($apiResponse,true);
        }

        return [];
    }

    private function createParcel($order, $bookNumber, $category, $carrierCode, $quantity, $czechlogisticOrder)
    {
        $apiKey = $this->scopeConfig->getValue('czechlogistic_config/general/apiKey', \Magento\Store\Model\ScopeInterface::SCOPE_STORE);
        $apiSecret = $this->scopeConfig->getValue('czechlogistic_config/general/apiSecret', \Magento\Store\Model\ScopeInterface::SCOPE_STORE);
        $ch = curl_init();

        $fields = array(
            'nr_ksiazki' => $bookNumber,
            'rodzaj' => $category,
            'kodpocztowy' => $order->getShippingAddress()->getPostcode(),
            'nazwaodbiorcy' => $order->getShippingAddress()->getCompany() . ' ' . $order->getShippingAddress()->getFirstname() . ' ' . $order->getShippingAddress()->getLastname(),
            'ulica' => implode(' ', $order->getShippingAddress()->getStreet()),
            'nrbudynku' => '',
            'miejscowosc' => $order->getShippingAddress()->getCity(),
            'email' => $order->getShippingAddress()->getEmail(),
            'nrtelefonu' => $order->getShippingAddress()->getTelephone(),
            'waga' => round(($order->getWeight()/$quantity), 1),
            'przewoznik' => $carrierCode,
            'doc_wz' => $order->getId(),
            'referencje' => $order->getId(),
            'ilosc' => $quantity
        );

        if (!empty($order->getClPickupPoint())) {
            $pickupPointId = $order->getClPickupPoint();
            $fields['rodzaj'] = 'point';

            $isNaPostuOrBalik = false;
            if (strpos($order->getClPickupPoint(), 'cp_napostu') !== false) {
                $pickupPointId = str_replace('cp_napostu', '', $pickupPointId);
                $isNaPostuOrBalik = true;
                $fields['rodzaj'] = 'baliknapostu';
            }
            if (strpos($order->getClPickupPoint(), 'cp_balikovna') !== false) {
                $pickupPointId = str_replace('cp_balikovna', '', $pickupPointId);
                $isNaPostuOrBalik = true;
                $fields['rodzaj'] = 'balikdobalikovny';
            }

            if ($carrierCode === 'kurier poczta') {
                $fields['miejscowosc'] = $czechlogisticOrder->getClPickupPointCity();
                $fields['kodpocztowy'] = $czechlogisticOrder->getClPickupPointPostcode();
                if ($isNaPostuOrBalik) {
                    $fields['kodpocztowy'] = $pickupPointId;
                }
            }

            if ($carrierCode === 'econt') {
                $fields['office_id'] = $pickupPointId;
            }

            if ($isNaPostuOrBalik) {
                $fields['point_id'] = null;
            }   else {
                $fields['point_id'] = $pickupPointId;
            }
        }


        if ($order->getPayment()->getMethod() === 'cashondelivery') {
            $fields['pobranie'] = round((float) $order->getGrandTotal(), 2);
        }

        $postvars = '';
        foreach($fields as $key=>$value) {
            $postvars .= $key . "=" . $value . "&";
        }
        $url = $this->czechApiUrl . '/przesylki/dodaj';
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            "CL-API-KEY: $apiKey",
            "CL-API-TOKEN: $apiSecret"
        ]);
        curl_setopt($ch,CURLOPT_URL,$url);
        curl_setopt($ch,CURLOPT_POST, 1);                //0 for a get request
        curl_setopt($ch,CURLOPT_POSTFIELDS,$postvars);
        curl_setopt($ch,CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch,CURLOPT_CONNECTTIMEOUT ,3);
        curl_setopt($ch,CURLOPT_TIMEOUT, 20);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, FALSE);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, FALSE);
        $response = curl_exec($ch);
        return json_decode($response);
    }
}
